<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Domain Locking: Only allow the plugin on staging.bullionmoney.com.au.
$allowed_domain = 'gold-feed.net';
if ( isset($_SERVER['HTTP_HOST']) && strtolower($_SERVER['HTTP_HOST']) !== strtolower($allowed_domain) ) {
    // If in admin area, show a notice.
    if ( is_admin() ) {
        add_action( 'admin_notices', function() use ( $allowed_domain ) {
            echo '<div class="error"><p><strong>Precious Metals Pricing Plugin Disabled:</strong> This plugin is licensed for ' . esc_html($allowed_domain) . ' only.</p></div>';
        } );
    }
    return;
}

class PreciousMetalsPricing {
    private $api_key = 'APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca';
    private $metals = ['gold', 'silver', 'platinum', 'palladium'];
    private $debug_logs = [];

    public function __construct() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_get_current_prices', [$this, 'get_current_prices']);
        add_action('wp_ajax_nopriv_get_current_prices', [$this, 'get_current_prices']);
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('save_post', [$this, 'save_meta_box_data']);
        add_action('precious_metals_pricing_cron', [$this, 'update_all_product_prices']);
        add_filter('cron_schedules', [$this, 'add_cron_interval']);
        add_action('wp_loaded', [$this, 'force_cron_on_load']);

        if (!wp_next_scheduled('precious_metals_pricing_cron')) {
            wp_schedule_event(time(), 'every_five_minutes', 'precious_metals_pricing_cron');
            $this->log("Cron scheduled at " . date('Y-m-d H:i:s'));
        }
    }

    public function add_cron_interval($schedules) {
        $schedules['every_five_minutes'] = [
            'interval' => 60,
            'display'  => __('Every Five Minutes', 'precious-metals-pricing')
        ];
        return $schedules;
    }

    public function force_cron_on_load() {
        if (current_user_can('manage_options') && !isset($_GET['skip_cron'])) {
            $this->update_all_product_prices();
        }
    }

    private function log($message) {
        $this->debug_logs[] = "[Precious Metals Pricing] " . $message;
        set_transient('precious_metals_logs', $this->debug_logs, 3600);
    }

    public function enqueue_frontend_scripts() {
        if (is_shop() || is_product_category() || is_product_tag() || is_product() || is_cart() || is_checkout() || is_front_page() || is_home()) {
            wp_enqueue_script('refresh-prices', plugin_dir_url(__FILE__) . 'js/refresh-prices.js', ['jquery'], '1.7', true);
            $logs = get_transient('precious_metals_logs') ?: $this->debug_logs;
            wp_localize_script('refresh-prices', 'preciousMetalsRefresh', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('get_current_prices_nonce'),
                'currency' => get_woocommerce_currency(),
                'is_admin' => false,
                'debug_logs' => $logs
            ]);
            wp_add_inline_style('woocommerce-general', '
                .woocommerce-Price-amount:not(.price-loaded) { visibility: hidden; }
                .woocommerce-Price-amount.price-loaded { visibility: visible; }
                .precious-metals-buyback-price .buyback-price-value:not(.price-loaded) { visibility: hidden; }
                .precious-metals-buyback-price .buyback-price-value.price-loaded { visibility: visible; }
            ');
        }
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'post.php' || get_post_type() !== 'product') {
            return;
        }
        wp_enqueue_script('refresh-prices', plugin_dir_url(__FILE__) . 'js/refresh-prices.js', ['jquery'], '1.7', true);
        $logs = get_transient('precious_metals_logs') ?: $this->debug_logs;
        wp_localize_script('refresh-prices', 'preciousMetalsRefresh', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('get_current_prices_nonce'),
            'currency' => get_woocommerce_currency(),
            'is_admin' => true,
            'product_id' => get_the_ID(),
            'debug_logs' => $logs
        ]);
    }

    public function add_meta_box() {
        add_meta_box(
            'precious_metals_pricing_meta_box',
            'Precious Metals Pricing Settings',
            [$this, 'render_meta_box'],
            'product',
            'normal',
            'high'
        );
    }

    public function render_meta_box($post) {
        wp_nonce_field('precious_metals_pricing_nonce', 'precious_metals_pricing_nonce');
        $fields = $this->get_product_settings($post->ID);
        ?>
        <p><label for="_goldpricelive_metal">Metal:</label>
        <select name="_goldpricelive_metal" id="_goldpricelive_metal">
            <?php foreach ($this->metals as $metal): ?>
                <option value="<?php echo esc_attr($metal); ?>" <?php selected($fields['_goldpricelive_metal'], $metal); ?>><?php echo ucfirst($metal); ?></option>
            <?php endforeach; ?>
        </select></p>
        <p><label for="_goldpricelive_weight">Weight:</label>
        <input type="number" step="any" name="_goldpricelive_weight" id="_goldpricelive_weight" value="<?php echo esc_attr($fields['_goldpricelive_weight']); ?>"></p>
        <p><label>Units:</label><br>
            <input type="radio" name="_goldpricelive_units" value="toz" <?php checked($fields['_goldpricelive_units'], 'toz'); ?>> Troy Ounce<br>
            <input type="radio" name="_goldpricelive_units" value="g" <?php checked($fields['_goldpricelive_units'], 'g'); ?>> Gram<br>
            <input type="radio" name="_goldpricelive_units" value="pw" <?php checked($fields['_goldpricelive_units'], 'pw'); ?>> Pennyweight
        </p>
        <p><label for="_goldpricelive_purity">Purity:</label>
        <input type="number" step="0.001" name="_goldpricelive_purity" id="_goldpricelive_purity" value="<?php echo esc_attr($fields['_goldpricelive_purity']); ?>"></p>
        <p><label for="_goldpricelive_fixed_amount">Fixed Amount:</label>
        <input type="number" step="any" name="_goldpricelive_fixed_amount" id="_goldpricelive_fixed_amount" value="<?php echo esc_attr($fields['_goldpricelive_fixed_amount']); ?>"></p>
        <p><label for="_goldpricelive_markup">Markup (%):</label>
        <input type="number" step="any" name="_goldpricelive_markup" id="_goldpricelive_markup" value="<?php echo esc_attr($fields['_goldpricelive_markup']); ?>"></p>
        <p><label for="_goldpricelive_lowest">Lowest Price:</label>
        <input type="number" step="any" name="_goldpricelive_lowest" id="_goldpricelive_lowest" value="<?php echo esc_attr($fields['_goldpricelive_lowest']); ?>"></p>
        <p><label for="_goldpricelive_buy_back">Buy Back Discount (%):</label>
        <input type="number" step="any" name="_goldpricelive_buy_back" id="_goldpricelive_buy_back" value="<?php echo esc_attr($fields['_goldpricelive_buy_back']); ?>"></p>
        <p><strong>Current Regular Price:</strong> <span id="current-regular-price"><?php echo wc_price(get_post_meta($post->ID, '_regular_price', true)); ?></span></p>
        <p><strong>Current Buyback Price:</strong> <span id="current-buyback-price"><?php echo wc_price(get_post_meta($post->ID, '_buyback_price', true)); ?></span></p>
        <p><em>Tiered pricing is managed by the Tiered Price Table for WooCommerce plugin.</em></p>
        <?php
    }

    public function save_meta_box_data($post_id) {
        if (!isset($_POST['precious_metals_pricing_nonce']) || !wp_verify_nonce($_POST['precious_metals_pricing_nonce'], 'precious_metals_pricing_nonce')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;

        $fields = [
            '_goldpricelive_metal', '_goldpricelive_weight', '_goldpricelive_units', '_goldpricelive_purity',
            '_goldpricelive_fixed_amount', '_goldpricelive_markup', '_goldpricelive_lowest', '_goldpricelive_buy_back'
        ];
        foreach ($fields as $field) {
            update_post_meta($post_id, $field, isset($_POST[$field]) ? sanitize_text_field($_POST[$field]) : '');
        }

        $this->update_product_price($post_id);
        $this->log("Saved settings for Product ID {$post_id} - " . json_encode($this->get_product_settings($post_id)));
    }

    public function fetch_metal_price($metal, $currency) {
        $currency = strtolower($currency);
        $url = "https://livegoldfeed.com/paid/{$this->api_key}/all_metals_json_{$currency}.php";
        $response = wp_remote_get($url);
        if (is_wp_error($response)) {
            $this->log("Failed to fetch {$metal} price in {$currency} - " . $response->get_error_message());
            return 0;
        }
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        $price = isset($data["{$metal}_ask_{$currency}_toz"]) ? floatval($data["{$metal}_ask_{$currency}_toz"]) : 0;
        $this->log("Fetched {$metal} price in {$currency} = {$price}");
        return $price;
    }

    private function calculate_metal_adjustment($base_price, $settings) {
        $weight = floatval($settings['_goldpricelive_weight'] ?? 0);
        $purity = floatval($settings['_goldpricelive_purity'] ?? 1);
        $units = $settings['_goldpricelive_units'] ?? 'toz';
        $weight_in_oz = $units === 'g' ? $weight / 31.1035 : ($units === 'pw' ? $weight / 20 : $weight);
        $adjustment = $base_price * $weight_in_oz * $purity;
        $this->log("Metal adjustment = {$adjustment} (Base: {$base_price}, Weight: {$weight}, Units: {$units}, Purity: {$purity})");
        return $adjustment;
    }

    private function calculate_price($metal_adjustment, $settings) {
        $fixed_amount = floatval($settings['_goldpricelive_fixed_amount'] ?? 0);
        $markup = floatval($settings['_goldpricelive_markup'] ?? 0);
        $lowest = floatval($settings['_goldpricelive_lowest'] ?? 0);
        $price = $metal_adjustment + $fixed_amount;
        $price += $price * ($markup / 100);
        $final_price = max($price, $lowest);
        $this->log("Calculated price = {$final_price} (Adjustment: {$metal_adjustment}, Fixed: {$fixed_amount}, Markup: {$markup}, Lowest: {$lowest})");
        return $final_price;
    }

    private function calculate_buyback_price($total_price, $settings) {
        $buyback_discount = floatval($settings['_goldpricelive_buy_back'] ?? 0);
        $buyback = $total_price * (1 - $buyback_discount / 100);
        $this->log("Buyback price = {$buyback} (Total: {$total_price}, Discount: {$buyback_discount})");
        return $buyback;
    }

    private function get_product_settings($product_id) {
        return [
            '_goldpricelive_metal'         => get_post_meta($product_id, '_goldpricelive_metal', true) ?: 'gold',
            '_goldpricelive_weight'        => get_post_meta($product_id, '_goldpricelive_weight', true),
            '_goldpricelive_units'         => get_post_meta($product_id, '_goldpricelive_units', true) ?: 'toz',
            '_goldpricelive_purity'        => get_post_meta($product_id, '_goldpricelive_purity', true),
            '_goldpricelive_fixed_amount'  => get_post_meta($product_id, '_goldpricelive_fixed_amount', true),
            '_goldpricelive_markup'        => get_post_meta($product_id, '_goldpricelive_markup', true),
            '_goldpricelive_lowest'        => get_post_meta($product_id, '_goldpricelive_lowest', true),
            '_goldpricelive_buy_back'      => get_post_meta($product_id, '_goldpricelive_buy_back', true)
        ];
    }

    public function update_product_price($product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            $this->log("Failed to load product ID {$product_id}");
            return;
        }

        $settings = $this->get_product_settings($product_id);
        $this->log("Updating Product ID {$product_id} - Settings: " . json_encode($settings));

        if (floatval($settings['_goldpricelive_weight']) <= 0 || floatval($settings['_goldpricelive_purity']) <= 0) {
            $this->log("Skipping update for Product ID {$product_id} - Invalid weight or purity");
            return;
        }

        $base_currency   = get_woocommerce_currency();
        $current_price   = $this->fetch_metal_price($settings['_goldpricelive_metal'], $base_currency);
        $metal_adjustment = $this->calculate_metal_adjustment($current_price, $settings);
        $calculated_price = $this->calculate_price($metal_adjustment, $settings);
        $buyback_price    = $this->calculate_buyback_price($calculated_price, $settings);

        $product->set_regular_price($calculated_price);
        $product->set_price($calculated_price);
        $product->save();
        update_post_meta($product_id, '_buyback_price', $buyback_price);
        wc_delete_product_transients($product_id);

        $this->log("Updated Product ID {$product_id} - Regular Price: {$calculated_price}, Buyback: {$buyback_price}");
    }

    public function update_all_product_prices() {
        $this->log("Cron started at " . date('Y-m-d H:i:s'));
        $args = [
            'post_type' => 'product',
            'posts_per_page' => -1,
            'post_status' => 'publish',
            'fields' => 'ids'
        ];
        $product_ids = get_posts($args);
        $this->log("Found " . count($product_ids) . " products to update");

        foreach ($product_ids as $product_id) {
            $this->update_product_price($product_id);
        }

        $this->log("Cron completed at " . date('Y-m-d H:i:s'));
    }

    public function get_current_prices() {
        check_ajax_referer('get_current_prices_nonce', 'nonce');
        $product_ids = isset($_POST['product_ids']) ? array_map('intval', (array)$_POST['product_ids']) : [];
        $tiered_rules = isset($_POST['tiered_rules']) ? (array)$_POST['tiered_rules'] : []; // Accept tiered rules from JS
        $currency = sanitize_text_field($_POST['currency'] ?? get_woocommerce_currency());
        $response_data = [];

        foreach ($product_ids as $product_id) {
            $product = wc_get_product($product_id);
            if (!$product) {
                $this->log("Product ID {$product_id} - Not found");
                continue;
            }

            // Get the regular price (which may have been updated by the cron or save)
            $regular_price = floatval($product->get_regular_price('edit'));
            $buyback_price = floatval(get_post_meta($product_id, '_buyback_price', true));

            // Log the initial regular price
            $this->log("Product ID {$product_id} - Initial Regular Price: {$regular_price}, Buyback Price: {$buyback_price}");

            // Fetch tiered pricing rules: prioritize frontend-provided rules, fallback to meta
            $rules = isset($tiered_rules[$product_id]) && !empty($tiered_rules[$product_id]) ? $tiered_rules[$product_id] : get_post_meta($product_id, '_tiered_price_rules', true);
            $type = get_post_meta($product_id, '_tiered_price_rules_type', true) ?: 'percentage'; // Default to percentage if not set
            $this->log("Product ID {$product_id} - Tiered Price Rules: " . json_encode($rules) . ", Type: {$type}");

            // Calculate tiered prices based on the updated regular price
            $tiered_prices = [];
            if ($rules && is_array($rules)) {
                if ($type === 'percentage') {
                    foreach ($rules as $quantity => $discount) {
                        $quantity = strval($quantity); // Ensure quantity is a string to match HTML
                        $discount = floatval($discount);
                        $tiered_price = $regular_price * (1 - $discount / 100);
                        $tiered_prices[$quantity] = $tiered_price;
                        $this->log("Product ID {$product_id} - Quantity {$quantity}: Discount {$discount}%, Tiered Price {$tiered_price}");
                    }
                } elseif ($type === 'fixed') {
                    foreach ($rules as $quantity => $price) {
                        $quantity = strval($quantity);
                        $tiered_prices[$quantity] = floatval($price);
                        $this->log("Product ID {$product_id} - Quantity {$quantity}: Fixed Price {$price}");
                    }
                }
            }

            // Always include the default tier (quantity 1) with the regular price
            if (!isset($tiered_prices['1'])) {
                $tiered_prices['1'] = $regular_price;
                $this->log("Product ID {$product_id} - Added default tier (quantity 1): {$regular_price}");
            }

            // Apply currency conversion
            $converted_regular_price = apply_filters('yay_currency_convert_price', $regular_price);
            $converted_buyback_price = $buyback_price ? apply_filters('yay_currency_convert_price', $buyback_price) : 0;
            $converted_tiered_prices = [];
            foreach ($tiered_prices as $quantity => $price) {
                $converted_tiered_prices[$quantity] = apply_filters('yay_currency_convert_price', $price);
            }

            $response_data[$product_id] = [
                'regular_price' => number_format($converted_regular_price, 2, '.', ''),
                'buyback_price' => number_format($converted_buyback_price, 2, '.', ''),
                'tiered_prices' => array_map(function($price) { return number_format($price, 2, '.', ''); }, $converted_tiered_prices),
                'currency' => $currency
            ];
            $this->log("Product ID {$product_id} - Final response: " . json_encode($response_data[$product_id]));
        }

        $this->log("AJAX response for " . json_encode($product_ids) . " - " . json_encode($response_data));
        wp_send_json_success($response_data);
    }
}

new PreciousMetalsPricing();
