<?php

function theme_enqueue_styles() {
    wp_enqueue_style( 'child-style', get_stylesheet_directory_uri() . '/style.css', [] );
}
add_action( 'wp_enqueue_scripts', 'theme_enqueue_styles', 20 );

function avada_lang_setup() {
	$lang = get_stylesheet_directory() . '/languages';
	load_child_theme_textdomain( 'Avada', $lang );
}
add_action( 'after_setup_theme', 'avada_lang_setup' );

function ubermenu_cart_count() {
    if (class_exists('WooCommerce')) {
        $cart_count = WC()->cart->get_cart_contents_count();
        $cart_url = wc_get_cart_url();

        return '<a href="' . esc_url($cart_url) . '" class="cart-button">
            <i class="fa-solid fa-cart-shopping"></i> 
            <span class="cart-text">Cart</span> 
            <span class="cart-count">' . esc_html($cart_count) . '</span>
        </a>';
    }
    return '';
}
add_shortcode('ubermenu_cart', 'ubermenu_cart_count');


function add_new_product_badge() {
    global $product;

    if (!$product) {
        return;
    }

    // Get product publish date
    $post_date = get_the_date('Y-m-d', $product->get_id());
    $days_since_published = (strtotime(date('Y-m-d')) - strtotime($post_date)) / (60 * 60 * 24);

    // Show badge only if the product is less than 30 days old
    if ($days_since_published <= 900) {
        echo '<span class="new-product-badge"></span>';
    }
}

// ✅ Add badge at the very beginning of each product box in the shop loop
add_action('woocommerce_before_shop_loop_item', 'add_new_product_badge', 5);

// ✅ Add badge to the single product page
add_action('woocommerce_before_single_product_summary', 'add_new_product_badge', 5);




// ✅ Ensure it applies to related products and upsells
function is_related_product() {
    return did_action('woocommerce_after_shop_loop_item_title') > 0;
}

function is_upsell_product() {
    return did_action('woocommerce_product_meta_end') > 0;
}

// Force display of "Out Of Stock" button for out-of-stock products
function custom_wc_show_out_of_stock_button() {
    global $product;

    if (!$product->is_in_stock()) {
        echo '<a href="#" class="button wc-out-of-stock-button" style="color: red !important;">
        <i class="fa-solid fa-cart-shopping"></i> ' . esc_html__('Out Of Stock', 'woocommerce') . '
      </a>';
    }
}
// Show on shop page
add_action('woocommerce_after_shop_loop_item', 'custom_wc_show_out_of_stock_button', 10);

// Show on single product page
add_action('woocommerce_single_product_summary', 'custom_wc_show_out_of_stock_button', 30);

// Shortcode to display the 3 most recent blog posts
function display_recent_blog_posts() {
    $query = new WP_Query(array(
        'posts_per_page' => 3, // Get 3 latest posts
        'post_status'    => 'publish',
        'orderby'        => 'date',
        'order'          => 'DESC',
    ));

    if ($query->have_posts()) {
        $output = '<div class="recent-posts-wrapper">';

        while ($query->have_posts()) {
            $query->the_post();

            // Get featured image URL or fallback image
            $image_url = get_the_post_thumbnail_url(get_the_ID(), 'thumbnail');
            if (!$image_url) {
                $image_url = 'https://via.placeholder.com/100'; // Fallback placeholder image
            }

            // Get the excerpt and force [...] replacement
            $excerpt = wp_trim_words(get_the_excerpt(), 20);
            $excerpt = preg_replace('/\[\.\.\.\]/', '…', $excerpt); // Replace [...] with …

            $output .= '
            <div class="recent-post-item">
                <div class="recent-post-thumbnail">
                    <img src="' . esc_url($image_url) . '" alt="' . esc_attr(get_the_title()) . '">
                </div>
                <div class="recent-post-content">
                    <h3><a href="' . esc_url(get_permalink()) . '">' . esc_html(get_the_title()) . '</a></h3>
                    <p>' . esc_html($excerpt) . '</p>
                    <a class="read-more" href="' . esc_url(get_permalink()) . '">Read More >></a>
                </div>
            </div>';
        }

        $output .= '</div>';
        wp_reset_postdata();
    } else {
        $output = '<p>No recent posts found.</p>';
    }

    return $output;
}
add_shortcode('recent_posts', 'display_recent_blog_posts');

// Force WordPress to replace [...] with …
function custom_excerpt_more($more) {
    return '…'; // Replaces [...] with …
}
add_filter('excerpt_more', 'custom_excerpt_more');




function spot_price_ticker() {
    $api_urls = array(
        'EUR' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_eur.php',
        'AUD' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_aud.php',
        'USD' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_usd.php'
    );

    $currency = class_exists('WooCommerce') ? get_woocommerce_currency() : 'USD';
    $api_url = isset($api_urls[$currency]) ? $api_urls[$currency] : $api_urls['USD'];

    $response = wp_remote_get($api_url);
    if (is_wp_error($response)) {
        return '<p>Error fetching spot prices.</p>';
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (!$data) {
        return '<p>Error parsing data.</p>';
    }

    $currency_key = strtolower($currency);
    $prices = array(
        'gold'     => array('name' => 'GOLD', 'price' => $data["gold_ask_{$currency_key}_toz"] ?? '0.00', 'change' => $data["gold_change_dollar_{$currency_key}_toz"] ?? '0.00'),
        'silver'   => array('name' => 'SILVER', 'price' => $data["silver_ask_{$currency_key}_toz"] ?? '0.00', 'change' => $data["silver_change_dollar_{$currency_key}_toz"] ?? '0.00'),
        'platinum' => array('name' => 'PLATINUM', 'price' => $data["platinum_ask_{$currency_key}_toz"] ?? '0.00', 'change' => $data["platinum_change_dollar_{$currency_key}_toz"] ?? '0.00'),
        'palladium' => array('name' => 'PALLADIUM', 'price' => $data["palladium_ask_{$currency_key}_toz"] ?? '0.00', 'change' => $data["palladium_change_dollar_{$currency_key}_toz"] ?? '0.00')
    );

    $output = '<div class="spot-price-ticker" id="spot-price-ticker" data-currency="' . esc_attr($currency) . '" data-api-url="' . esc_url($api_url) . '">';
    foreach ($prices as $metal) {
        $change_color = ($metal['change'] < 0) ? 'red' : 'green';
        $arrow = ($metal['change'] < 0) ? '▼' : '▲';
        $formatted_price = number_format((float) $metal['price'], 2, '.', ',');
        $formatted_change = number_format((float) $metal['change'], 2, '.', ',');

        $output .= '<div class="spot-price-item">
            <strong>' . esc_html($metal['name']) . '</strong> 
            ' . esc_html(get_woocommerce_currency_symbol()) . '<span class="price-value">' . esc_html($formatted_price) . '</span> 
            <span class="change ' . esc_attr($change_color) . '">' . esc_html($arrow . ' ' . $formatted_change) . '</span>
        </div>';
    }
    $output .= '</div>';

    // Add JavaScript only once per page
    if (!wp_script_is('spot-price-sync', 'enqueued')) {
        wp_enqueue_script('spot-price-sync', '', [], null, true);
        $output .= '<script type="text/javascript" id="spot-price-sync">
            // Shared price update function
            window.updateSpotPrices = async function() {
                // Update ticker
                const ticker = document.getElementById("spot-price-ticker");
                if (ticker) {
                    const apiUrl = ticker.getAttribute("data-api-url");
                    const currency = ticker.getAttribute("data-currency").toLowerCase();
                    try {
                        const response = await fetch(apiUrl);
                        const data = await response.json();
                        
                        const items = ticker.getElementsByClassName("spot-price-item");
                        const metals = ["gold", "silver", "platinum", "palladium"];
                        
                        metals.forEach((metal, index) => {
                            const price = parseFloat(data[`${metal}_ask_${currency}_toz`]).toLocaleString("en-US", {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            const change = parseFloat(data[`${metal}_change_dollar_${currency}_toz`]);
                            items[index].querySelector(".price-value").textContent = price;
                            items[index].querySelector(".change").textContent = (change < 0 ? "▼ " : "▲ ") + change.toLocaleString("en-US", {minimumFractionDigits: 2, maximumFractionDigits: 2});
                            items[index].querySelector(".change").className = "change " + (change < 0 ? "red" : "green");
                        });
                    } catch (error) {
                        console.error("Ticker update error:", error);
                    }
                }

                // Trigger table update if present
                if (typeof window.fetchSpotPrices === "function") {
                    window.fetchSpotPrices();
                }
            };

            // Run on page load
            document.addEventListener("DOMContentLoaded", () => {
                window.updateSpotPrices();
                setInterval(window.updateSpotPrices, 30000); // Every 30 seconds
            });
        </script>';
    }

    return $output;
}
add_shortcode('spot_price_ticker', 'spot_price_ticker');


function reset_comment_count() {
    global $wpdb;
    $wpdb->query("UPDATE $wpdb->posts SET comment_count = 0");
    wp_cache_flush();
}
add_action('init', 'reset_comment_count');

// Filtered WooCommerce Categories by Metal Type
function avada_filtered_woo_categories() {
    if (!is_product_category() && !is_shop()) {
        return ''; // Run on category and shop pages
    }

    $current_category = get_queried_object();
    $current_category_name = is_product_category() ? $current_category->name : '';
    $current_url = add_query_arg(NULL, NULL); // Get current URL with query args

    // Detect the metal type from the current category
    $metals = array('Gold', 'Silver', 'Platinum', 'Palladium');
    $current_metal = '';

    if ($current_category_name) {
        foreach ($metals as $metal) {
            if (stripos($current_category_name, $metal) !== false) {
                $current_metal = $metal;
                break;
            }
        }
    }

    if (is_product_category() && !$current_metal) {
        return ''; // No metal detected on category page, don't show anything
    }

    $args = array(
        'taxonomy'   => 'product_cat',
        'orderby'    => 'name',
        'hide_empty' => true,
    );

    $categories = get_categories($args);
    $output = '<ul class="custom-category-list">';

    foreach ($categories as $category) {
        if ($category->count > 0 && (!$current_metal || stripos($category->name, $current_metal) !== false)) {
            $filter_url = add_query_arg(array('product_cat' => $category->slug), get_permalink(wc_get_page_id('shop')));
            $is_active = (isset($_GET['product_cat']) && $_GET['product_cat'] === $category->slug);
            $clear_url = remove_query_arg('product_cat', $current_url);

            $output .= '<li>';
            $output .= '<span class="filter-item">';
            $output .= '<a href="' . esc_url($filter_url) . '" class="filter-link">› ' . esc_html($category->name) . ' (' . esc_html($category->count) . ')</a>';
            if ($is_active) {
                $output .= ' <a href="' . esc_url($clear_url) . '" class="clear-filter">X</a>';
            }
            $output .= '</span>';
            $output .= '</li>';
        }
    }

    $output .= '</ul>';

    return $output;
}
add_shortcode('woo_filtered_categories', 'avada_filtered_woo_categories');

// Filtered Price Ranges
function avada_filtered_price_ranges() {
    if (!is_product_category() && !is_shop()) {
        return ''; // Only run on category and shop pages
    }

    $current_category = get_queried_object();
    $current_category_name = is_product_category() ? $current_category->name : '';
    $current_url = add_query_arg(NULL, NULL); // Get current URL with query args

    // Detect metal type
    $metals = array('Gold', 'Silver', 'Platinum', 'Palladium');
    $current_metal = '';

    if ($current_category_name) {
        foreach ($metals as $metal) {
            if (stripos($current_category_name, $metal) !== false) {
                $current_metal = $metal;
                break;
            }
        }
    }

    if (is_product_category() && !$current_metal) {
        return ''; // No metal detected on category page, don't show anything
    }

    $base_url = get_permalink(wc_get_page_id('shop'));
    $currency_symbol = get_woocommerce_currency_symbol();

    $price_ranges = array(
        '0-99.99'     => array('min' => 0, 'max' => 99.99),
        '100-499.99'  => array('min' => 100, 'max' => 499.99),
        '500-999.99'  => array('min' => 500, 'max' => 999.99),
        '1000-2499.99' => array('min' => 1000, 'max' => 2499.99),
        '2500-4999.99' => array('min' => 2500, 'max' => 4999.99),
        '5000-9999.99' => array('min' => 5000, 'max' => 9999.99),
        '10000-above' => array('min' => 10000, 'max' => 999999),
    );

    $product_counts = array_fill_keys(array_keys($price_ranges), 0);

    $args = array(
        'post_type'      => 'product',
        'posts_per_page' => -1,
        'meta_key'       => '_price',
        'orderby'        => 'meta_value_num',
        'order'          => 'ASC',
    );

    if (is_product_category()) {
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'product_cat',
                'field'    => 'name',
                'terms'    => $current_category_name,
            ),
        );
    }

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $product = wc_get_product(get_the_ID());
            $price = (float) $product->get_price();

            foreach ($price_ranges as $range_label => $range) {
                if ($price >= $range['min'] && $price <= $range['max']) {
                    $product_counts[$range_label]++;
                }
            }
        }
        wp_reset_postdata();
    }

    $output = '<ul class="custom-category-list">';
    foreach ($price_ranges as $range_label => $range) {
        if ($product_counts[$range_label] > 0) { // Only show if count > 0
            $formatted_min = number_format($range['min'], 2);
            $formatted_max = number_format($range['max'], 2);

            if ($range_label === '10000-above') {
                $price_text = "{$currency_symbol}10,000.00 and above";
            } else {
                $price_text = "{$currency_symbol}{$formatted_min} - {$currency_symbol}{$formatted_max}";
            }

            $filter_url = add_query_arg(array(
                'min_price' => $range['min'],
                'max_price' => $range['max'],
            ), $base_url);

            $is_active = (isset($_GET['min_price']) && isset($_GET['max_price']) && 
                          floatval($_GET['min_price']) == $range['min'] && 
                          floatval($_GET['max_price']) == $range['max']);
            $clear_url = remove_query_arg(array('min_price', 'max_price'), $current_url);

            $output .= '<li>';
            $output .= '<span class="filter-item">';
            $output .= '<a href="' . esc_url($filter_url) . '" class="filter-link">› ' . esc_html($price_text) . ' (' . esc_html($product_counts[$range_label]) . ')</a>';
            if ($is_active) {
                $output .= ' <a href="' . esc_url($clear_url) . '" class="clear-filter">X</a>';
            }
            $output .= '</span>';
            $output .= '</li>';
        }
    }
    $output .= '</ul>';

    return $output;
}
add_shortcode('woo_filtered_price', 'avada_filtered_price_ranges');

// Filtered Attributes (Condition, Mint/Brand, Composition)
function avada_filtered_attribute($attribute_name) {
    if (!is_product_category() && !is_shop()) {
        return ''; // Only run on category and shop pages
    }

    $current_category = get_queried_object();
    $current_category_name = is_product_category() ? $current_category->name : '';
    $current_url = add_query_arg(NULL, NULL); // Get current URL with query args

    // Detect metal type
    $metals = array('Gold', 'Silver', 'Platinum', 'Palladium');
    $current_metal = '';

    if ($current_category_name) {
        foreach ($metals as $metal) {
            if (stripos($current_category_name, $metal) !== false) {
                $current_metal = $metal;
                break;
            }
        }
    }

    if (is_product_category() && !$current_metal) {
        return ''; // No metal detected on category page, don't show anything
    }

    $base_url = get_permalink(wc_get_page_id('shop'));

    $attribute_values = array();
    $args = array(
        'post_type'      => 'product',
        'posts_per_page' => -1,
    );

    if (is_product_category()) {
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'product_cat',
                'field'    => 'name',
                'terms'    => $current_category_name,
            ),
        );
    }

    $query = new WP_Query($args);

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();
            $product = wc_get_product(get_the_ID());
            $attribute_value = $product->get_attribute($attribute_name);

            // For Composition, filter by current metal type
            if ($attribute_name === 'Composition' && $current_metal && !empty($attribute_value)) {
                if (stripos($attribute_value, $current_metal) === false) {
                    continue; // Skip if the composition doesn't match the current metal
                }
            }

            if (!empty($attribute_value)) {
                if (!isset($attribute_values[$attribute_value])) {
                    $attribute_values[$attribute_value] = 0;
                }
                $attribute_values[$attribute_value]++;
            }
        }
        wp_reset_postdata();
    }

    $attribute_slug = sanitize_title($attribute_name); // e.g., "mint-brand" from "Mint/Brand"
    $query_param = "filter_{$attribute_slug}";
    $output = '<ul class="custom-category-list">';
    foreach ($attribute_values as $value => $count) {
        if ($count > 0) { // Only show if count > 0
            $filter_value = sanitize_title($value); // e.g., "british-royal-mint"
            $filter_url = add_query_arg(array($query_param => $filter_value), $base_url);
            $is_active = (isset($_GET[$query_param]) && $_GET[$query_param] === $filter_value);
            $clear_url = remove_query_arg($query_param, $current_url);

            $output .= '<li>';
            $output .= '<span class="filter-item">';
            $output .= '<a href="' . esc_url($filter_url) . '" class="filter-link">› ' . esc_html($value) . ' (' . esc_html($count) . ')</a>';
            if ($is_active) {
                $output .= ' <a href="' . esc_url($clear_url) . '" class="clear-filter">X</a>';
            }
            $output .= '</span>';
            $output .= '</li>';
        }
    }
    $output .= '</ul>';

    return $output;
}

add_shortcode('woo_filtered_condition', function() { return avada_filtered_attribute('Condition'); });
add_shortcode('woo_filtered_mint_brand', function() { return avada_filtered_attribute('Mint/Brand'); });
add_shortcode('woo_filtered_composition', function() { return avada_filtered_attribute('Composition'); });

// Clear Filters Button
function avada_clear_filters() {
    $shop_url = get_permalink(wc_get_page_id('shop'));

    // Remove all WooCommerce filter query vars
    $shop_url = remove_query_arg(array(
        'product_cat',
        'min_price',
        'max_price',
        'filter_condition',
        'filter_mint-brand',
        'filter_composition',
    ), $shop_url);

    return '<div class="clear-filters">
                <a href="' . esc_url($shop_url) . '" class="fusion-button button-flat fusion-button-default-size button-default fusion-button-default">Clear All Filters</a>
            </div>';
}
add_shortcode('woo_clear_filters', 'avada_clear_filters');