<?php
/**
 * Plugin Name: Gold-Feed.com WooCommerce Custom Price Table
 * Plugin URI: https://gold-feed.com.com
 * Description: Allows admin to create multiple product groups and display them using shortcodes with search functionality, AJAX price refreshing, and multiple product selection/removal.
 * Version: 2.1
 * Author: BS
 * License: GPL2
 */

if (!defined('ABSPATH')) {
    exit;
}

// 1. Add Admin Menu
function cpt_admin_menu() {
    add_options_page(
        'Product Price Table Settings',
        'Product Price Table',
        'manage_options',
        'custom-price-table',
        'cpt_admin_page'
    );
}
add_action('admin_menu', 'cpt_admin_menu');

// 2. Enqueue Scripts
function cpt_enqueue_admin_scripts($hook) {
    if ($hook !== 'settings_page_custom-price-table') {
        return;
    }
    
    wp_enqueue_script('jquery');
    wp_localize_script('jquery', 'cptAjax', [
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('cpt_product_search')
    ]);
}
add_action('admin_enqueue_scripts', 'cpt_enqueue_admin_scripts');

// 3. Admin Settings Page
function cpt_admin_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html__('Precious Metals Product Price Table Settings', 'cpt'); ?></h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('cpt_settings_group');
            do_settings_sections('custom-price-table');
            submit_button();
            ?>
        </form>

        <style type="text/css">
            .product-group {
                margin: 10px 0;
                padding: 10px;
                border: 1px solid #ddd;
                background: #fff;
            }
            .product-group input[type="text"] {
                width: 200px;
                margin-right: 10px;
            }
            .product-list-container {
                margin-top: 10px;
                max-height: 200px;
                overflow-y: auto;
            }
            .product-item {
                padding: 5px;
                border-bottom: 1px solid #eee;
                display: flex;
                align-items: center;
            }
            .product-item:last-child {
                border-bottom: none;
            }
            .remove-product {
                color: red;
                cursor: pointer;
                margin-left: 10px;
                font-weight: bold;
            }
            .search-results {
                margin-top: 5px;
            }
        </style>

        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Add new group
            $('#add-group').on('click', function() {
                let groupNames = new Set();
                $('.product-group input[name^="cpt_product_groups"][name$="[name]"]').each(function() {
                    groupNames.add($(this).val().trim().toLowerCase());
                });

                let newGroupName = prompt('Enter a unique group name:');
                if (!newGroupName || groupNames.has(newGroupName.trim().toLowerCase())) {
                    alert('Group name must be unique and non-empty!');
                    return;
                }

                let randomID = 'group_' + Math.floor(Math.random() * 100000);
                let template = `
                    <div class="product-group">
                        <input type="text" name="cpt_product_groups[${randomID}][name]" value="${newGroupName}" placeholder="Group Name">
                        <input type="text" class="search-product" placeholder="Search Product by Name or ID">
                        <div class="product-list-container" data-group-id="${randomID}"></div>
                        <button type="button" class="button remove-group">Remove Group</button>
                    </div>
                `;
                $('#product-groups-container').append(template);
            });

            // Remove group
            $(document).on('click', '.remove-group', function() {
                $(this).closest('.product-group').remove();
            });

            // Product search and add
            $(document).on('keyup', '.search-product', function() {
                let $input = $(this);
                let $container = $input.next('.product-list-container');
                let groupId = $container.data('group-id');
                let search = $input.val();

                if (search.length < 2) {
                    $container.find('.search-results').remove();
                    return;
                }

                $.ajax({
                    url: cptAjax.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'cpt_search_products',
                        search: search,
                        nonce: cptAjax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            let existingIds = [];
                            $container.find('input[type="checkbox"]').each(function() {
                                existingIds.push($(this).val());
                            });

                            let options = '<div class="search-results">';
                            response.data.forEach(product => {
                                if (!existingIds.includes(product.id.toString())) {
                                    options += `
                                        <div class="product-item">
                                            <input type="checkbox" name="cpt_product_groups[${groupId}][products][]" value="${product.id}">
                                            ${product.name}
                                            <span class="remove-product" title="Remove Product">✕</span>
                                        </div>`;
                                }
                            });
                            options += '</div>';

                            $container.find('.search-results').remove();
                            $container.append(options);
                        }
                    }
                });
            });

            // Remove product
            $(document).on('click', '.remove-product', function() {
                $(this).closest('.product-item').remove();
            });
        });
        </script>
    </div>
    <?php
}

// 4. Register Settings
function cpt_register_settings() {
    register_setting('cpt_settings_group', 'cpt_product_groups', 'cpt_sanitize_callback');
    add_settings_section('cpt_main_section', 'Manage Product Groups', null, 'custom-price-table');
    add_settings_field('cpt_product_groups_field', 'Product Groups', 'cpt_product_groups_field_cb', 'custom-price-table', 'cpt_main_section');
}
add_action('admin_init', 'cpt_register_settings');

// 5. Sanitize Callback
function cpt_sanitize_callback($input) {
    if (!is_array($input)) return [];
    $sanitized_groups = [];
    foreach ($input as $key => $group) {
        if (!isset($group['name']) || empty($group['name'])) continue;
        $group_key = sanitize_key($group['name']);
        $sanitized_groups[$group_key] = [
            'name' => sanitize_text_field($group['name']),
            'products' => array_filter(array_map('intval', (array)($group['products'] ?? [])))
        ];
    }
    return $sanitized_groups;
}

// 6. Display Groups in Admin
function cpt_product_groups_field_cb() {
    $groups = get_option('cpt_product_groups', []);
    ?>
    <div id="product-groups-container">
        <?php
        if (!empty($groups)) {
            foreach ($groups as $group_key => $group_data) {
                echo cpt_generate_group_html($group_key, $group_data['name'], $group_data['products']);
            }
        }
        ?>
    </div>
    <button type="button" class="button" id="add-group"><?php esc_html_e('+ Add Group', 'cpt'); ?></button>
    <?php
}

// 7. Generate Group HTML
function cpt_generate_group_html($group_key, $group_name, $product_ids) {
    $html = '<div class="product-group">';
    $html .= sprintf(
        '<input type="text" name="cpt_product_groups[%s][name]" value="%s" placeholder="%s">',
        esc_attr($group_key),
        esc_attr($group_name),
        esc_attr__('Group Name', 'cpt')
    );
    $html .= sprintf('<input type="text" class="search-product" placeholder="%s">', 
        esc_attr__('Search Product by Name or ID', 'cpt')
    );
    $html .= sprintf('<div class="product-list-container" data-group-id="%s">', esc_attr($group_key));
    $html .= cpt_get_products_list($group_key, $product_ids);
    $html .= '</div>';
    $html .= sprintf('<button type="button" class="button remove-group">%s</button>', 
        esc_html__('Remove Group', 'cpt')
    );
    $html .= '</div>';
    return $html;
}

// 8. Get Products List
function cpt_get_products_list($group_key, $selected_ids) {
    $html = '';
    if (!empty($selected_ids)) {
        foreach ($selected_ids as $product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
                $html .= sprintf(
                    '<div class="product-item"><input type="checkbox" name="cpt_product_groups[%s][products][]" value="%d" checked>%s<span class="remove-product" title="Remove Product">✕</span></div>',
                    esc_attr($group_key),
                    $product_id,
                    esc_html($product->get_name())
                );
            }
        }
    }
    return $html;
}

// 9. AJAX Product Search
function cpt_search_products() {
    check_ajax_referer('cpt_product_search', 'nonce');
    
    $search = sanitize_text_field($_POST['search']);
    $args = [
        'post_type' => 'product',
        'posts_per_page' => 10,
        's' => $search,
        'post_status' => 'publish'
    ];
    
    $products = [];
    $query = new WP_Query($args);
    while ($query->have_posts()) {
        $query->the_post();
        $product = wc_get_product(get_the_ID());
        $products[] = [
            'id' => get_the_ID(),
            'name' => $product->get_name()
        ];
    }
    wp_reset_postdata();
    
    wp_send_json_success($products);
}
add_action('wp_ajax_cpt_search_products', 'cpt_search_products');

// 10. AJAX Handler for Price Refresh
function cpt_refresh_prices() {
    check_ajax_referer('cpt_price_refresh', 'nonce');
    
    $group_key = sanitize_title($_POST['group']);
    $groups = get_option('cpt_product_groups', []);
    
    if (empty($groups[$group_key])) {
        wp_send_json_error('Group not found');
    }

    $product_ids = $groups[$group_key]['products'] ?? [];
    if (empty($product_ids)) {
        wp_send_json_error('No products in group');
    }

    $prices = [];
    foreach ($product_ids as $product_id) {
        $product = wc_get_product($product_id);
        if (!$product) continue;
        
        $sell_price = $product->get_price();
        $buyback_percentage = floatval(get_post_meta($product_id, '_goldpricelive_buy_back', true));
        
        if ($buyback_percentage > 0) {
            $buyback_price = $sell_price - ($sell_price * ($buyback_percentage / 100));
        } else {
            $buyback_price = 'N/A';
        }
        
        if ($buyback_price !== 'N/A' && function_exists('yay_currency_convert_price')) {
            $buyback_price = apply_filters('yay_currency_convert_price', $buyback_price);
        }
        
        $prices[$product_id] = [
            'sell' => wc_price($sell_price),
            'buyback' => ($buyback_price !== 'N/A') ? wc_price($buyback_price) : $buyback_price
        ];
    }
    
    wp_send_json_success($prices);
}
add_action('wp_ajax_cpt_refresh_prices', 'cpt_refresh_prices');
add_action('wp_ajax_nopriv_cpt_refresh_prices', 'cpt_refresh_prices');

// 11. Shortcode to Display Products
function cpt_display_price_table($atts) {
    $atts = shortcode_atts(['group' => ''], $atts, 'custom_price_table');
    $groups = get_option('cpt_product_groups', []);
    $group_key = sanitize_title($atts['group']);

    if (empty($groups[$group_key])) {
        return sprintf('<p>%s <strong>%s</strong>.</p>', 
            esc_html__('No products at this time. Please check back:', 'cpt'),
            esc_html($atts['group'])
        );
    }

    $product_ids = $groups[$group_key]['products'] ?? [];
    if (empty($product_ids)) {
        return '<p>' . esc_html__('No products in this group.', 'cpt') . '</p>';
    }

    wp_enqueue_script('jquery');
    wp_localize_script('jquery', 'cptPriceAjax', [
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('cpt_price_refresh'),
        'group' => $group_key
    ]);

    ob_start();
    ?>
    <div class="cpt-price-table-wrapper">
        <table class="product-price-table" data-group="<?php echo esc_attr($group_key); ?>">
            <thead>
                <tr>
                    <th><?php esc_html_e('Product', 'cpt'); ?></th>
                    <th><?php esc_html_e('Selling Price', 'cpt'); ?></th>
                    <th><?php esc_html_e('Buy Back', 'cpt'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php 
                foreach ($product_ids as $product_id):
                    $product = wc_get_product($product_id);
                    if (!$product) continue;
                    
                    $sell_price = $product->get_price();
                    $buyback_percentage = floatval(get_post_meta($product_id, '_goldpricelive_buy_back', true));
                    
                    if ($buyback_percentage > 0) {
                        $buyback_price = $sell_price - ($sell_price * ($buyback_percentage / 100));
                    } else {
                        $buyback_price = 'N/A';
                    }
                    
                    if ($buyback_price !== 'N/A' && function_exists('yay_currency_convert_price')) {
                        $buyback_price = apply_filters('yay_currency_convert_price', $buyback_price);
                    }
                ?>
                    <tr data-product-id="<?php echo esc_attr($product_id); ?>">
                        <td><a href="<?php echo esc_url(get_permalink($product_id)); ?>" title="<?php echo esc_attr($product->get_name()); ?>"><?php echo esc_html($product->get_name()); ?></a></td>
                        <td class="sell-price"><?php echo wc_price($sell_price); ?></td>
                        <td class="buyback-price"><?php echo ($buyback_price !== 'N/A') ? wc_price($buyback_price) : esc_html($buyback_price); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

   

    <script type="text/javascript">
    jQuery(document).ready(function($) {
        function refreshPrices() {
            const $table = $('.product-price-table[data-group="<?php echo esc_js($group_key); ?>"]');
            const group = $table.data('group');
            
            $.ajax({
                url: cptPriceAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'cpt_refresh_prices',
                    group: group,
                    nonce: cptPriceAjax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $.each(response.data, function(productId, prices) {
                            const $row = $table.find('tr[data-product-id="' + productId + '"]');
                            $row.find('.sell-price').html(prices.sell);
                            $row.find('.buyback-price').html(prices.buyback);
                        });
                    }
                },
                error: function() {
                    console.log('Price refresh failed');
                }
            });
        }

        // Refresh every 60 seconds (1 minute)
        setInterval(refreshPrices, 60000);
        
        // Initial refresh after 5 seconds
        setTimeout(refreshPrices, 5000);
    });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('custom_price_table', 'cpt_display_price_table');

// 12. Print Button Shortcode
function cpt_print_button_shortcode() {
    wp_enqueue_script('jquery');
    ob_start();
    ?>
    <button class="cpt-print-button"><?php esc_html_e('Print Live Prices', 'cpt'); ?></button>

    <style type="text/css">
        .cpt-print-button {
            padding: 10px 20px;
            background-color: #0073aa;
            color: #fff;
            border: none;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            margin: 10px 0;
            transition: background-color 0.3s ease;
        }
        .cpt-print-button:hover {
            background-color: #005d87;
        }
        @media print {
            .cpt-print-button {
                display: none;
            }
        }
    </style>

    <script type="text/javascript">
    jQuery(document).ready(function($) {
        $('.cpt-print-button').on('click', function() {
            window.print();
        });
    });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('cpt_print_button', 'cpt_print_button_shortcode');

// 13. Load Text Domain
function cpt_load_textdomain() {
    load_plugin_textdomain('cpt', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'cpt_load_textdomain');

// 14. Enqueue Styles
function cpt_enqueue_styles() {
    wp_enqueue_style('cpt-styles', plugin_dir_url(__FILE__) . 'css/style.css');
}
add_action('wp_enqueue_scripts', 'cpt_enqueue_styles');