<?php
/**
 * Plugin Name: Gold-Feed.com Spot Price Table
 * Description: Displays a live spot price table with the store's selected currency.
 * Version: 1.9.2
 * Author: BS
 */

if (!defined('ABSPATH')) {
    exit;
}

function spot_price_table_shortcode() {
    if (!class_exists('WooCommerce')) {
        return '<p>WooCommerce is required for this plugin to work.</p>';
    }

    $current_currency = get_woocommerce_currency();
    ob_start(); ?>

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" integrity="sha512-DTOQO9RWCH3ppGqcWaEA1BIZOC6xxalwEsw9c2QQeAIftl+Vegovlnee1c9QX4TctnWMn13TZye+giMm8e2LwA==" crossorigin="anonymous" referrerpolicy="no-referrer" />

    <div class="spot-price-container" data-currency="<?php echo esc_attr($current_currency); ?>">
        <table width="100%" cellspacing="0" cellpadding="5">
            <thead>
                <tr>
                    <th>Metal</th>
                    <th><?php echo esc_html($current_currency); ?> (1oz)</th>
                    <th>USD (1oz)</th>
                </tr>
            </thead>
            <tbody class="spot-price-data">
                <tr><td colspan="3">Loading prices...</td></tr>
            </tbody>
        </table>
        <p class="last-updated">
            Last updated: <span class="last-updated-time">-</span>
            <a href="https://bullionmoney.com.au/charts/gold/" class="homesublnk">More Detail ></a>
        </p>
    </div>

    <script>
    window.fetchSpotPrices = async function() {
        const containers = document.querySelectorAll('.spot-price-container');
        containers.forEach(async (container) => {
            const currency = container.getAttribute('data-currency');
            const dataBody = container.querySelector('.spot-price-data');
            const lastUpdated = container.querySelector('.last-updated-time');

            try {
                // Fetch prices based on selected currency
                let responseSelected, dataSelected;
                if (currency === 'AUD') {
                    responseSelected = await fetch('https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_aud.php');
                    dataSelected = await responseSelected.json();
                } else if (currency === 'EUR') {
                    responseSelected = await fetch('https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_eur.php');
                    dataSelected = await responseSelected.json();
                } else { // Default to USD
                    responseSelected = await fetch('https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_usd.php');
                    dataSelected = await responseSelected.json();
                }

                // Always fetch USD prices for the second column
                const responseUSD = await fetch('https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_usd.php');
                const dataUSD = await responseUSD.json();

                // Define metals with prices and changes
                const currencyKey = currency.toLowerCase();
                const metals = [
                    { 
                        name: 'Gold', 
                        selected: parseFloat(dataSelected[`gold_ask_${currencyKey}_toz`]), 
                        usd: parseFloat(dataUSD.gold_ask_usd_toz),
                        selectedChange: parseFloat(dataSelected[`gold_change_dollar_${currencyKey}_toz`]),
                        usdChange: parseFloat(dataUSD.gold_change_dollar_usd_toz)
                    },
                    { 
                        name: 'Silver', 
                        selected: parseFloat(dataSelected[`silver_ask_${currencyKey}_toz`]), 
                        usd: parseFloat(dataUSD.silver_ask_usd_toz),
                        selectedChange: parseFloat(dataSelected[`silver_change_dollar_${currencyKey}_toz`]),
                        usdChange: parseFloat(dataUSD.silver_change_dollar_usd_toz)
                    },
                    { 
                        name: 'Platinum', 
                        selected: parseFloat(dataSelected[`platinum_ask_${currencyKey}_toz`]), 
                        usd: parseFloat(dataUSD.platinum_ask_usd_toz),
                        selectedChange: parseFloat(dataSelected[`platinum_change_dollar_${currencyKey}_toz`]),
                        usdChange: parseFloat(dataUSD.platinum_change_dollar_usd_toz)
                    },
                    { 
                        name: 'Palladium', 
                        selected: parseFloat(dataSelected[`palladium_ask_${currencyKey}_toz`]), 
                        usd: parseFloat(dataUSD.palladium_ask_usd_toz),
                        selectedChange: parseFloat(dataSelected[`palladium_change_dollar_${currencyKey}_toz`]),
                        usdChange: parseFloat(dataUSD.palladium_change_dollar_usd_toz)
                    },
                ];

                // Set currency symbol
                const currencySymbol = currency === 'EUR' ? '€' : '$';

                let tableRows = '';
                metals.forEach(metal => {
                    const selectedPriceFormatted = metal.selected.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
                    const usdPrice = metal.usd.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
                    const selectedClass = getPriceChangeClass(metal.selectedChange);
                    const usdClass = getPriceChangeClass(metal.usdChange);
                    const selectedArrow = getArrow(metal.selectedChange);
                    const usdArrow = getArrow(metal.usdChange);

                    tableRows += `<tr>
                        <td><strong>${metal.name}</strong></td>
                        <td class="${selectedClass}" data-metal="${metal.name}" data-currency="${currency}">
                            <span class="price-text">${currencySymbol}${selectedPriceFormatted}</span> ${selectedArrow}
                        </td>
                        <td class="${usdClass}" data-metal="${metal.name}" data-currency="USD">
                            <span class="price-text">$${usdPrice}</span> ${usdArrow}
                        </td>
                    </tr>`;
                });

                dataBody.innerHTML = tableRows;
                lastUpdated.innerText = new Date().toLocaleTimeString();

                setTimeout(() => {
                    container.querySelectorAll('.price-up, .price-down').forEach(el => {
                        el.classList.remove('price-up', 'price-down');
                        el.classList.add('fade-to-black');
                        const arrow = el.querySelector('.arrow-up, .arrow-down');
                        if (arrow) {
                            arrow.classList.add('fade-to-black');
                            arrow.classList.remove('arrow-up', 'arrow-down');
                            arrow.classList.add('arrow-neutral');
                        }
                    });
                }, 3000);

            } catch (error) {
                dataBody.innerHTML = '<tr><td colspan="3">Error fetching prices</td></tr>';
                console.error('Error fetching prices:', error);
            }
        });
    };

    function getPriceChangeClass(change) {
        if (change > 0) return 'price-up';
        if (change < 0) return 'price-down';
        return 'price-neutral';
    }

    function getArrow(change) {
        if (change > 0) return '<i class="fa-solid fa-angle-up arrow-up"></i>';
        if (change < 0) return '<i class="fa-solid fa-angle-down arrow-down"></i>';
        return '<i class="fa-solid fa-minus arrow-neutral"></i>';
    }

    // Trigger the fetch immediately for all instances
    window.fetchSpotPrices();
    </script>

    <style>
        .spot-price-container {
            font-family: 'Roboto', sans-serif;
            font-size: 14px;
            position: relative;
        }
        table {
            border-collapse: collapse;
            width: 100%;
            background: #fff;
            font-size: 14px;
            border: 2px solid #ddd;
            border-radius: 2px;
            overflow: hidden;
        }
        th {
            background: black;
            color: white;
            padding: 5px;
            text-align: left;
            border: none;
            font-weight: bold;
        }
        td {
            padding: 5px;
            border: none;
        }
        tr:nth-child(odd) {
            background-color: #f3f3f3;
        }
        .last-updated {
            padding-top: 10px;
            margin-top: 10px;
            font-size: 14px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .homesublnk {
            text-decoration: none;
            color: #0d538e;
            font-weight: bold;
        }
        .homesublnk:hover {
            text-decoration: underline;
        }
        .price-up {
            color: green;
            animation: flashGreen 0.5s;
        }
        .price-down {
            color: red;
            animation: flashRed 0.5s;
        }
        .price-neutral {
            color: black;
        }
        .arrow-up {
            font-size: 14px;
            margin-left: 5px;
            color: green;
            font-weight: 900;
        }
        .arrow-down {
            font-size: 14px;
            margin-left: 5px;
            color: red;
            font-weight: 900;
        }
        .arrow-neutral {
            font-size: 14px;
            margin-left: 5px;
            color: black;
            font-weight: 900;
        }
        .fade-to-black {
            transition: color 1s ease-in-out;
            color: black !important;
        }
        .price-text {
            transition: color 1s ease-in-out;
            font-weight: bold;
        }
        @keyframes flashGreen {
            0% { background-color: #d4ffd4; }
            100% { background-color: transparent; }
        }
        @keyframes flashRed {
            0% { background-color: #ffd4d4; }
            100% { background-color: transparent; }
        }
    </style>

    <?php
    return ob_get_clean();
}

add_shortcode('spot_price_table', 'spot_price_table_shortcode');