<?php
/**
 * Plugin Name: Gold-Feed.com Precious Metals Price Charts
 * Plugin URI: https://gold-feed.com
 * Description: Displays gold, silver, platinum, and palladium price charts using a shortcode. Syncs with WooCommerce's active currency. Loads chart with real-time ask prices.
 * Version: 4.0
 * Author: BS
 * Author URI: https://gold-feed.com
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Enqueue scripts
function precious_metals_chart_enqueue_scripts() {
    if (!is_admin()) {
        wp_enqueue_script('jquery');
        wp_enqueue_script('highcharts', 'https://code.highcharts.com/stock/highstock.js', ['jquery'], null, true);
        wp_enqueue_script('highcharts-exporting', 'https://code.highcharts.com/modules/exporting.js', ['highcharts'], null, true);
        
        $currency = class_exists('WooCommerce') ? get_woocommerce_currency() : 'USD';
        
        wp_localize_script('jquery', 'preciousMetalsChart', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('currency_update'),
            'currency' => $currency,
            'liveFeedUrls' => [
                'USD' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_usd.php',
                'AUD' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_aud.php',
                'EUR' => 'https://livegoldfeed.com/paid/APIcb4f8f3e66a1df36a70192b0255760092be1185e4658bf4cca/all_metals_json_eur.php'
            ]
        ]);
    }
}
add_action('wp_enqueue_scripts', 'precious_metals_chart_enqueue_scripts');

// Shortcode function to display the chart
function display_precious_metals_chart($atts) {
    $atts = shortcode_atts(['metal' => 'gold'], $atts);
    
    $metal = strtolower($atts['metal']);
    $valid_metals = ['gold', 'silver', 'platinum', 'palladium'];
    if (!in_array($metal, $valid_metals)) {
        return '<p style="color: red;">Invalid metal type. Use gold, silver, platinum, or palladium.</p>';
    }

    $chart_id = 'chart_' . uniqid();

    ob_start();
    ?>
    <div id="<?= esc_attr($chart_id); ?>" style="height: 500px; width:100%;"></div>
    <div id="chart-error-<?= esc_attr($chart_id); ?>" style="color: red; display: none;"></div>

    <script>
        (function($) {
            function getWooCurrency() {
                let currency = preciousMetalsChart.currency;
                console.log("WooCommerce currency detected: " + currency);
                return ["AUD", "EUR", "USD"].includes(currency) ? currency : 'USD';
            }

            function fetchLivePrice(metal, currency, callback) {
                let feedUrl = preciousMetalsChart.liveFeedUrls[currency];
                if (!feedUrl) {
                    console.error("No live feed URL for currency: " + currency);
                    callback(null, null);
                    return;
                }
                // Add timestamp to prevent caching
                feedUrl += (feedUrl.includes('?') ? '&' : '?') + 't=' + new Date().getTime();
                $.getJSON(feedUrl, function(data) {
                    const metalKey = metal + '_ask_' + currency.toLowerCase() + '_toz';
                    const changeKey = metal + '_change_dollar_' + currency.toLowerCase() + '_toz';
                    let livePrice = data[metalKey];
                    let priceChange = data[changeKey];
                    if (!livePrice || priceChange === undefined) {
                        console.error("No ask price or change found for " + metalKey + " in data:", data);
                        callback(null, null);
                        return;
                    }
                    console.log("Fetched live ask price for " + metal + " in " + currency + ": " + livePrice + ", Change: " + priceChange);
                    callback(parseFloat(livePrice), parseFloat(priceChange));
                }).fail(function(jqXHR, textStatus, errorThrown) {
                    console.error("Failed to fetch live price from " + feedUrl + ": " + textStatus, errorThrown);
                    callback(null, null);
                });
            }

            function updateChart_<?= esc_js($chart_id); ?>(currency) {
                console.log("Updating chart with currency: " + currency);
                var metal = "<?= esc_js($metal); ?>";
                var chartUrls = {
                    gold: {
                        AUD: "https://goldwindowtrading.com/charting/gold_prices_aud.php",
                        EUR: "https://goldwindowtrading.com/charting/gold_prices_eur.php",
                        USD: "https://goldwindowtrading.com/charting/gold_prices_usd.php"
                    },
                    silver: {
                        AUD: "https://goldwindowtrading.com/charting/silver_prices_aud.php",
                        EUR: "https://goldwindowtrading.com/charting/silver_prices_eur.php",
                        USD: "https://goldwindowtrading.com/charting/silver_prices_usd.php"
                    },
                    platinum: {
                        AUD: "https://goldwindowtrading.com/charting/platinum_prices_aud.php",
                        EUR: "https://goldwindowtrading.com/charting/platinum_prices_eur.php",
                        USD: "https://goldwindowtrading.com/charting/platinum_prices_usd.php"
                    },
                    palladium: {
                        AUD: "https://goldwindowtrading.com/charting/palladium_prices_aud.php",
                        EUR: "https://goldwindowtrading.com/charting/palladium_prices_eur.php",
                        USD: "https://goldwindowtrading.com/charting/palladium_prices_usd.php"
                    }
                };

                if (!["AUD", "EUR", "USD"].includes(currency)) {
                    console.warn("Unsupported currency: " + currency + ", defaulting to USD");
                    currency = 'USD';
                }

                var chartUrl = chartUrls[metal][currency];

                if (!chartUrl) {
                    console.error("Chart URL not found for " + metal + " in " + currency);
                    $("#chart-error-<?= esc_js($chart_id); ?>").text("Error: Chart URL missing for " + currency).show();
                    return;
                }

                console.log("Fetching chart data from: " + chartUrl);

                $.getJSON(chartUrl, function(data) {
                    console.log("Fetched chart data for " + currency, data);

                    if (!data || data.length === 0) {
                        console.warn("No data received for " + metal + " chart in " + currency + ". Rendering empty chart.");
                        data = [];
                    }

                    // Initial chart render
                    fetchLivePrice(metal, currency, function(livePrice, priceChange) {
                        let currentDate = new Date().toLocaleDateString();
                        let priceToShow = livePrice !== null ? livePrice : 'N/A';
                        let lineColor = priceChange > 0 ? '#0da311' : priceChange < 0 ? '#FF0000' : '#000000';

                        console.log("Initial render with livePrice: " + livePrice + ", Change: " + priceChange);

                        var chart = Highcharts.stockChart("<?= esc_js($chart_id); ?>", {
                            chart: { backgroundColor: "#ffffff" },
                            exporting: { enabled: false },
                            rangeSelector: {
                                selected: 1,
                                buttons: [
                                    { type: "month", count: 1, text: "1m" },
                                    { type: "month", count: 3, text: "3m" },
                                    { type: "year", count: 1, text: "1y" },
                                    { type: "all", text: "All" }
                                ],
                                inputEnabled: false
                            },
                            yAxis: {
                                title: { text: "Price per t/oz" },
                                min: livePrice ? Math.min(...data.map(d => d[1]), livePrice) * 0.95 : null,
                                max: livePrice ? Math.max(...data.map(d => d[1]), livePrice) * 1.05 : null,
                                plotLines: livePrice !== null ? [{
                                    id: 'live-price-line',
                                    value: livePrice,
                                    color: lineColor,
                                    dashStyle: "solid",
                                    width: 2,
                                    zIndex: 5,
                                    label: { 
                                        text: "Live Ask Price (" + currentDate + "): " + livePrice.toFixed(2) + " " + currency,
                                        align: 'left',
                                        style: { color: lineColor, fontWeight: 'normal' }
                                    }
                                }] : []
                            },
                            series: [{
                                name: currency + " " + metal.toUpperCase() + " Close Price",
                                data: data,
                                tooltip: { valueDecimals: 2 },
                                color: metal === "gold" ? "#e6d12d" : 
                                       metal === "silver" ? "#C0C0C0" : 
                                       metal === "platinum" ? "#B0B0B0" : "#A0A0A0"
                            }]
                        });
                        $("#chart-error-<?= esc_js($chart_id); ?>").hide();

                        // Update live price every 30 seconds
                        setInterval(function() {
                            fetchLivePrice(metal, currency, function(newLivePrice, newPriceChange) {
                                if (newLivePrice !== null) {
                                    let newDate = new Date().toLocaleDateString();
                                    let newLineColor = newPriceChange > 0 ? '#0da311' : newPriceChange < 0 ? '#FF0000' : '#000000';
                                    
                                    console.log("Updating live price to: " + newLivePrice + ", Change: " + newPriceChange);

                                    chart.yAxis[0].removePlotLine('live-price-line');
                                    chart.yAxis[0].addPlotLine({
                                        id: 'live-price-line',
                                        value: newLivePrice,
                                        color: newLineColor,
                                        dashStyle: "solid",
                                        width: 2,
                                        zIndex: 5,
                                        label: { 
                                            text: "Live Ask Price (" + newDate + "): " + newLivePrice.toFixed(2) + " " + currency,
                                            align: 'left',
                                            style: { color: newLineColor, fontWeight: 'normal' }
                                        }
                                    });
                                } else {
                                    console.warn("No valid live price received for update.");
                                }
                            });
                        }, 30000); // Update every 30 seconds to match ticker
                    });
                }).fail(function(jqXHR, textStatus, errorThrown) {
                    console.error("Failed to load JSON data for " + currency, textStatus, errorThrown);
                    $("#chart-error-<?= esc_js($chart_id); ?>").text("Error loading data for " + currency).show();
                });
            }

            function syncWithWooCommerce() {
                let currency = getWooCurrency();
                console.log("Syncing with WooCommerce currency: " + currency);
                updateChart_<?= esc_js($chart_id); ?>(currency);
            }

            $(document).ready(function() {
                console.log("Initializing chart sync...");
                setTimeout(syncWithWooCommerce, 1000);

                let lastCurrency = getWooCurrency();
                setInterval(function() {
                    var currentCurrency = getWooCurrency();
                    if (currentCurrency !== lastCurrency) {
                        console.log("WooCommerce currency changed from " + lastCurrency + " to " + currentCurrency);
                        lastCurrency = currentCurrency;
                        updateChart_<?= esc_js($chart_id); ?>(currentCurrency);
                    }
                }, 5000);
            });
        })(jQuery);
    </script>
    <?php
    return ob_get_clean();
}

add_shortcode('precious_metals_chart', 'display_precious_metals_chart');